/*=============================================================================

LCD_LIB_de	Funktionen fuer die LCD-Anzeige des MiniMEXLE
=========================================================

Dateiname: lcd_lib_de.h

Bibliothek fuer das LCD-Display des MiniMexle Boards bzw. optimiert fuer LCD-Display HD44780

Autor: 		Steffen Freihofer (2006)
Adaption: 	Thomas Pospiech (2007, Timing-Probleme geloest)
			G. Gruhler, D. Chilachava (2009, Init Ports und Dokumentation)
			Peter Blinzinger Libary umgebaut fuer SimulIDE, kompatibel mit dem Mexle2020 Board

Version: 	1.3	vom 26.04.2020

=============================================================================*/


// Deklarationen ==============================================================

// Festlegung der Quarzfrequenz

#ifndef F_CPU					// optional definieren
#define F_CPU 12288000UL		// MiniMEXLE mit 12,288 MHz Quarz
#endif


// Include von Header-Dateien

#include <avr/io.h>			// I/O Konfiguration (intern weitere Dateien)
#include <avr/interrupt.h>	// globale Interrupts definieren (sei() und cli())
#include <stdint.h>			// Definition von Typen (int, char, ...)
#include <stdbool.h>		// Definition von 1-Bit-Variablentypen (bool)
#include <util/delay.h>		// Definition von Delays (Wartezeiten)


// Konstanten

#define CLEAR_DISPLAY	1	// Instruction Code fuer LCD: Loeschen des Displays


// Port-Bits

#define E PD4				// Enable-Signal zum Display: Port D, PD4
#define RS PD7				// Register-Select zum Display: Port D, PD5


// Makros
#define Datenregister	DDRD //Datenrichtungsregister  fuer Datenport zum LCD (Verwendung der Bits 0 bis 3)
#define Steuerregister	DDRD //Datenrichtungsregister  fuer Steuerport zum LCD
#define P_DATA 	  PORTD		// Port ist Datenport zum LCD (Verwendung der Bits 0 bis 3)
#define P_STEUER  PORTD		// Port ist Steuerport zum LCD


// Funktionsprototypen

void lcd_enable (void);
void lcd_write (unsigned char byte);
void lcd_init (void);
void lcd_putc (unsigned char zeichen);
void lcd_putstr (char *string);
void lcd_gotoxy (unsigned char line, unsigned char pos);
void lcd_clearDisplay(void);
void lcd_clearline (unsigned char line);
void lcd_displayMessage(char *string, unsigned char, unsigned char);


// LCD-Ansteuerfunktionen =====================================================

// ----------------------------------------------------------------------------
// LCD_ENABLE:		Erzeugt einen HIGH-Impuls des Enable-Signals
// ----------------------------------------------------------------------------

void lcd_enable (void)
{
	P_STEUER |= (1 << E);		// E=1   1  +-+
	_delay_us (1);
	P_STEUER &= ~(1 << E);		// E=0   0--+ +---
}


// ----------------------------------------------------------------------------
// LCD_WRITE:	Schreibt ein Byte im 4-Bit-Modus in das LCD
// ----------------------------------------------------------------------------

void lcd_write (unsigned char byte)	// Ausgabewert 8 Bit in "byte"
{
	unsigned char temp;		// Definition lokale Variable "temp"

	temp = byte;			// Ausgabewert zwischenspeichern

	P_DATA &= 0xF0;			// Untere 4 Bit auf Datenport loeschen
	P_DATA |= (temp>>4);	// Ausgabewert 4 Bit nach rechts schieben
							//    oberes Nibble auf Daten-Port LCD schreiben

	lcd_enable();			// Erzeugung Enable-Impuls zur Datenspeicherung
							// keine Wartezeit zwischen Nibbles notwendig

	byte &= 0x0F;			// Unteres Nibble Ausgabewert extrahieren

	P_DATA &= 0xF0;			// Untere 4 Bit auf Datenport loeschen
	P_DATA |= byte;			// unteres Nibble auf Daten-Port LCD schreiben

	lcd_enable();			// Erzeugung Enable-Impuls zur Datenspeicherung
}


// ----------------------------------------------------------------------------
// LCD_INIT:	Initialisiert das LCD 
// ----------------------------------------------------------------------------

void lcd_init (void)
{
	cli();						//globale Interrupts deaktivieren

	Datenregister |= 0x0f;					// Bit 0..3 (LCD-Daten) auf Output
	Steuerregister |= ((1<<E) | (1<<RS));	// Bit RS: LCD Register Select und Bit E: LCD Enable auf Output
	P_DATA |= 0x0f;							// Port, Bit0..3 (LCD-Daten) SET 

	// Steuersignale auf LOW
	P_STEUER &= ~((1<<E) | (1<<RS));	// E und RS auf 0 setzen
	_delay_ms (50);

	// Display in 4-bit Modus initialisieren
	P_DATA &= 0xF0;
	P_DATA |= 0x03;
	lcd_enable();
	_delay_ms (10);
	lcd_enable ();
	_delay_ms (10);
	lcd_enable ();
		
	P_DATA &= 0xF0;//0xF1;
	P_DATA |=0x02;
	_delay_ms (2);
	lcd_enable();
	_delay_us (50);
	
	// 2 Zeilen, 5 x 8 Pixel
	lcd_write (0x28);
	_delay_us (40);
	
	// Display einschalten
	lcd_write (0x0C);
	_delay_us (50);
	
	//P_DATA &= 0xF1;
	//lcd_enable();
	//_delay_us (50);
	
	// Cursor auf "increment mode" schalten
	lcd_write (0x06);//(0x02);
	_delay_us (50);

	//	Display loeschen (funktioniert nur, wenn es 2x aufgerufen wird)
	lcd_write (0x01);
	_delay_ms (3);
	lcd_write (0x01);
	_delay_ms (3);

	sei();						// globale Interrrupts aktivieren
}


// ----------------------------------------------------------------------------
// LCD_PUTC:	Schreibt ein Zeichen an die entsprechende Speicherstelle
// ----------------------------------------------------------------------------

void lcd_putc (unsigned char character)	// Ausgabewert 8-Bit in "character"
{
	P_STEUER |= (1<<RS);		// Register Select auf HIGH: "Daten ausgeben"
	
	lcd_write (character);		// Zeichen ausgeben

	_delay_us (55);				// Verzoegerung 55 us (interne Verarbeitung)
}


// ----------------------------------------------------------------------------
// LCD_PUTSTR:	Schreibt einen String auf das LCD
// ----------------------------------------------------------------------------

void lcd_putstr (char *string)	// Pointer auf String in "string"
{
	while (*string)				// solange nicht Stringendezeichen 0x00
	{
		lcd_putc (*string);		// Zeichen aus String holen und ausgeben

		string ++;				// Pointer auf naechstes Zeichen setzen
	}
}


// ----------------------------------------------------------------------------
// LCD_GOTOXY:	Verschiebt den Curser an die angegebene Position
// ----------------------------------------------------------------------------

void lcd_gotoxy (unsigned char line, unsigned char pos)
{
	P_STEUER &= ~(1<<RS);		// Register Select auf LOW: "Control ausgeben"
		
	lcd_write((1 << 7) + 0x40 * line + pos); // Adresse Cursor in DDRAM-Adresse 

	_delay_us (50);				// Verzoegerung 50 us fuer Interne Verarbeitung				
}


// ----------------------------------------------------------------------------
// LCD_CLEARDISPLAY:	Loescht den gesamten Display-Inhalt
// ----------------------------------------------------------------------------

void lcd_clearDisplay(void)
{
	P_STEUER &= ~(1 << RS);		// Register Select auf LOW: "Control ausgeben"
	
	lcd_write(CLEAR_DISPLAY);	// Befehlscode an Display ausgeben

	_delay_ms (2);				// Verzoegerung 2 ms fuer interne Verarbeitung

}


// ----------------------------------------------------------------------------
// LCD_CLEARLINE:	Loescht den Inhalt einer Display-Zeile
// ----------------------------------------------------------------------------

void lcd_clearline (unsigned char line)
{
	switch(line)
	{
		case 0: 	lcd_gotoxy(0,0); // Beginn erste Zeile addressieren
					lcd_putstr ("                "); // Leerzeichen ausgeben
					break;

		case 1: 	lcd_gotoxy(1,0); // Beginn zweite Zeile addressieren 
					lcd_putstr ("                "); // Leerzeichen ausgeben
					break;
	}
}


// ----------------------------------------------------------------------------
// LCD_DISPLAYMESSAGE:	Anzeige einer Stringvariablen auf dem Display
// ----------------------------------------------------------------------------

void lcd_displayMessage(char* mess, unsigned char line, unsigned char pos)
{
	lcd_gotoxy(line,pos);		// Cursor positionieren
	lcd_putstr(mess);			// String ausgeben
}

