/* ----------------------------------------------------------------------------
 
 Experiment 10:   I2C Kommunikation
 =============    ===============================
 
 Dateiname	: I2C_SimpleMaster.c
  
 Autoren    : Tim Fischer       (Hochschule Heilbronn, Fakultaet T1)
               
 Datum      : 28.11.22
 
 Version    : 1.1
 
 Hardware:  Simulide 1.0.0-RC3 (R1148)
 
 Software:  Entwicklungsumgebung: AtmelStudio 7.0
            C-Compiler: AVR/GNU C Compiler 5.4.0
 
 Funktion : TBD
 
 Displayanzeige:    TBD
 
 Tastenfunktion:    keine
 
 Jumperstellung:    keine
 
 Fuses im uC:       CKDIV8: Aus (keine generelle Vorteilung des Takts)
 
  
// ----------------------------------------------------------------------------*/
 
// Deklarationen ==============================================================
 
// Festlegung der Quarzfrequenz
#define F_CPU 12288000UL	// CPU Frequenz von 12.288MHz
#define F_SCL 100000L		// Baudrate von 100 kHz

// Include von Header-Dateien
#include <stdbool.h>			// Definition von 1-Bit-Variablentypen (bool)
#include <avr/interrupt.h>
#include <util/delay.h>

// Konstanten
#define SET_BIT(PORT, BIT)  ((PORT) |=  (1 << (BIT))) // Port-Bit Zustand setzen
#define CLR_BIT(PORT, BIT)  ((PORT) &= ~(1 << (BIT))) // Port-Bit Zustand loeschen
#define TGL_BIT(PORT, BIT)  ((PORT) ^=  (1 << (BIT))) // Port-Bit Zustand wechseln (toggle)

#define PRESCALER_VAL       60      // Faktor Vorteiler = 60
#define CYCLE10MS_MAX       10      // Faktor Hundertstel = 10
#define CYCLE100MS_MAX      10      // Faktor Zehntel = 10

unsigned char softwarePrescaler = PRESCALER_VAL;    // Zaehlvariable Vorteiler
unsigned char cycle10msCount    = CYCLE10MS_MAX;    // Zaehlvariable Hundertstel
unsigned char cycle100msCount   = CYCLE100MS_MAX;   // Zaehlvariable Zehntel

bool timertick;                     // Bit-Botschaft alle 0,166ms (bei Timer-Interrupt)
bool cycle10msActive;               // Bit-Botschaft alle 10ms
bool cycle100msActive;              // Bit-Botschaft alle 100ms
bool cycle1sActive;                 // Bit-Botschaft alle 1s

const int   TEMP[45]    =  {521,499,479,459,440,422,404,388,371,354,
							338,323,308,293,279,264,250,236,221,207,
							193,179,165,151,137,122,108,93,78,63,
							48,32,15,-1,-19,-38,-56,-77,-97,-121,
							-145,-173,-202,-237,-278};
// Die Tabellenwerte sind in 1/10 C angegeben
// Der erste Tabellenwert entspricht einem AD-Wert
// von 256. Die Abstaende der AD-Werte sind 16

unsigned int    adcValue    = 0;    // Variable fuer den AD-Wandlungswert
int             tValue      = 0;    // Variable fuer die Temperatur (in 1/10 C)

uint8_t TWI_Address =  0b0001010;
uint8_t TWI_Register= 0b00000000;
uint8_t TWI_Data	= 0b00110111;

//Funktionsprototypen
void initTimer0(void);
void initAdc(void);
void doAdc(void);
void calculateTemp(void);

void sendTwiData(void);
void requestTwiData(void);
void initTwi(void);
void transmitTwiStart(void);
void transmitTwiDataOrAddress(char Data);
void transmitTwiStop(void);
uint8_t readTwiDataOrAddressAck(void);
uint8_t readTwiDataOrAddressNack(void);

int main(void)
{
    initTimer0();										// Initialisierung von Timer0
    initAdc();											// Initialisierung des AD-Wandlers
    DDRD=0xFF;
    sei();										        // generell Interrupts einschalten
	_delay_ms(1500);									// abwarten der Anzeige auf dem Slave
	
	while (1)
	{
        if(cycle100msActive)							// Durchfuehrung der Funktion einmal pro 100ms
        {
	        cycle100msActive = 0;							// Taktbotschaft zuruecksetzen
            doAdc();            // Ausfuehrung des Modules der A/D-Wandlung
            calculateTemp();    // Ausfuehrung des Modules der Umrechnung
			sendTwiData();
			_delay_us(1);	
			requestTwiData();
			_delay_us(1);			
		}
	}
}

// Timer Initialisierung ==============================================================
//
// Initialisierung des Timer0 zur Erzeugung eines getakteten Interrupts.
// Er dient dazu, die benoetigten Taktbotschaften zu erzeugen.
void initTimer0()
{
	TCCR0A  |= (0<<WGM00)
			|  (0<<WGM01);											// Timer 0 auf "Normal Mode" schalten
	TCCR0B  |= (0<<WGM02)
			|  (1<<CS01 );											// mit Prescaler /8 betreiben
	TIMSK0  |= (1<<TOIE0);											// Overflow-Interrupt aktivieren
}
 
 // ADC-Initialisierung ==============================================================
 //
 // Initialisierung des A/D-Wandlers:
 // Vorteiler = 128 => interner Takt = 96 kHz
 // Abfrage des ADC0 (NTC-Spannungsteiler)
 // Referenzspannung = analoge Versorgung Avcc
 void initAdc()
 {
	 ADMUX   |= (1<<REFS0);					// Vref =AVCC; ADC0
	 
	 ADCSRA  |= (1<<ADPS0)
			 |  (1<<ADPS1)
			 |  (1<<ADPS2)
			 |  (1<<ADEN);					// Teiler 128; ADC ON
	 (void) ADCH;							// erster Wert ist "wegzuwerfen"
 }
 
 // ADWandlung ==============================================================
 //
 // Durchfuehrung einer Einzelwandlung der am NTC-Spannungsteiler anstehenden
 // Spannung in einen digitalen 10-bit-Wert (einmal pro 100 ms).
 void doAdc()
 {
	 ADCSRA |= (1<<ADSC);					// Wandlung starten
	 while (ADCSRA & (1<<ADSC));			// Ende der Wandlung abwarten
	 
	 adcValue = ADCL + (ADCH<<8);			// 10-Bit-Wert berechnen
	 // ADCL muss vor ADCH stehen!!
	 // siehe Datenblatt des ATmega 328
 }
 
 // Umrechnung ==============================================================
 //
 // (wird alle 100 ms aufgerufen)
 void calculateTemp ()
 {
	 unsigned char index;                // Tabellenindex fuer Temperaturtabelle
	 unsigned char steps;                // Abstand zum naechstkleineren Wert
	 // der AD-Werte der Temperaturtabelle
	 
	 index   = (adcValue-256)/16;        // Indexberechnung (Zeiger in Tabelle)
	 steps   = (adcValue-256)%16;        // Rest fuer Tabellen-Interpolation
	 
	 tValue = steps * (TEMP[index+1] - TEMP[index])/16 + TEMP[index];
	 // Temperaturwert berechnen
}
 
 // Interrupt-Routine ==========================================================

ISR (TIMER0_OVF_vect)
/*  In der Interrupt-Routine sind die Softwareteiler realisiert, die die Takt-
    botschaften (10ms, 100ms, 1s) fuer die gesamte Uhr erzeugen. Die Interrupts
    werden von Timer 0 ausgeloest (Interrupt Nr. 1)
  
*/
{
    timertick = 1;							// Botschaft 0,166ms senden
    --softwarePrescaler;                    // Vorteiler dekrementieren
    if (softwarePrescaler==0)               // wenn 0 erreicht: 10ms abgelaufen
    {
        softwarePrescaler = PRESCALER_VAL;	//    Vorteiler auf Startwert
        cycle10msActive = 1;				//    Botschaft 10ms senden
        --cycle10msCount;					//    Hunderstelzaehler dekrementieren
  
        if (cycle10msCount==0)				// wenn 0 erreicht: 100ms abgelaufen
        {
            cycle10msCount = CYCLE10MS_MAX; // Teiler auf Startwert
            cycle100msActive = 1;			//    Botschaft 100ms senden
            --cycle100msCount;              //    Zehntelzaehler dekrementieren
  
            if (cycle100msCount==0)         // wenn 0 erreicht: 1s abgelaufen
            {
                cycle100msCount = CYCLE100MS_MAX;	//    Teiler auf Startwert
                cycle1sActive = 1;					//    Botschaft 1s senden
            }
        }
    }
}

// I2C Daten Senden ==============================================================
//
void sendTwiData()
{
	initTwi();										// Initialisierung von TWI anstoen
	transmitTwiStart();								// Startbit schreiben
	transmitTwiDataOrAddress((TWI_Address<<1) + 0);	// Adresse senden
	transmitTwiDataOrAddress(TWI_Register);			// Register senden
	transmitTwiDataOrAddress(tValue & 0xFF);		// Daten senden (Low Byte)
	transmitTwiDataOrAddress((char) (tValue >> 8));// Daten senden (High Byte)
	transmitTwiStop();								// Stoppbit schreiben
}

// I2C Daten Empfangen ==============================================================
//
void requestTwiData()
{
	initTwi();										// Initialisierung von TWI anstoen
	transmitTwiStart();								// Startbit schreiben
	transmitTwiDataOrAddress((TWI_Address<<1) + 1);	// Adresse senden
	PORTD=readTwiDataOrAddressNack();
	transmitTwiStop();								// Stoppbit schreiben
}

// I2C Initialisierung ==============================================================
//
void initTwi()
{
	TWSR = CLR_BIT(TWSR, TWPS0);// Es wird kein Prescaler verwendet
	TWSR = CLR_BIT(TWSR, TWPS1);//
	TWCR = 0;					// Control Register zurcksetzen
	TWBR = ((F_CPU/F_SCL)-16)/2;// die Bitrate wird mittels CPU Frequenz und Serial Clock Frequenz ermittelt
}

// I2C Startbit senden ==============================================================
//
void transmitTwiStart()
{
	TWCR = (1<<TWINT)|(1<<TWEN)|(1<<TWSTA); // TWSTA = Startbit aktivieren, TWEN = TWI starten (ENable), TWINT = Interrupt bit lschen (durch setzen)
	while (!(TWCR & (1<<TWINT)));			// warten bis bertragung erfolgreich, Trigger ist hier das Setzen von TWINT
}

// I2C Adressbyte/Daten senden ==============================================================
//
void transmitTwiDataOrAddress(char Data)						
{
	TWDR = Data;
	TWCR = (1<<TWINT)|(1<<TWEN);			// TWEN = TWI starten (ENable), TWINT = Interrupt bit lschen (durch setzen)
	while (!(TWCR & (1<<TWINT)));			// warten bis bertragung erfolgreich, Trigger ist hier das Setzen von TWINT
}

// I2C Stoppbit senden ==============================================================
//
void transmitTwiStop()
{
	TWCR=(1<<TWINT)|(1<<TWEN)|(1<<TWSTO);	// TWSTO = Stopptbit aktivieren, TWEN = TWI starten (ENable), TWINT = Interrupt bit lschen (durch setzen)
}

/*******************************************************
 Public Function: TWIM_ReadAck

 Purpose: Read a byte from the slave and request next byte

 Input Parameter: None

 Return Value: uint8_t
  	- uint8_t	Read byte

*******************************************************/
uint8_t readTwiDataOrAddressAck (void)
	{
	TWCR = (1<<TWINT)|(1<<TWEN)|(1<<TWEA);
	while (!(TWCR & (1<<TWINT)));    
	return TWDR;
	}
/*******************************************************
 Public Function: TWIM_ReadAck

 Purpose: Read the last byte from the slave

 Input Parameter: None

 Return Value: uint8_t
  	- uint8_t	Read byte

*******************************************************/
uint8_t readTwiDataOrAddressNack  (void)
	{
	TWCR = (1<<TWINT)|(1<<TWEN);
	while(!(TWCR & (1<<TWINT)));
	
	return TWDR;
	}
