/* ----------------------------------------------------------------------------
  
 Experiment 10:   I2C Kommunikation
 =============    ===============================
  
 Dateiname  : I2C_Slave.c
  
 Autoren    : Tim Fischer       (Hochschule Heilbronn, Fakultaet T1)
  
 Datum      : 28.11.22
  
 Version    : 1.1
   
 Hardware:  Simulide 1.0.0-RC3 (R1148)
  
 Software:  Entwicklungsumgebung: AtmelStudio 7.0
            C-Compiler: AVR/GNU C Compiler 5.4.0
  
 Funktion:  tbd
 
  
   
// ----------------------------------------------------------------------------*/
  
// Deklarationen ==============================================================
  
// Festlegung der Quarzfrequenz
#define F_CPU 12288000UL	// CPU Frequenz von 12.288MHz
 
// Include von Header-Dateien
#include <util/twi.h> 	    //enthlt z.B. die Bezeichnungen fr die Statuscodes in TWSR
#include <avr/interrupt.h>  //dient zur Behandlung der Interrupts
#include <stdint.h>         //definiert den Datentyp uint8_t
#include "lcd_lib_de.h"     // Header-Datei fuer LCD-Anzeige

// Konstanten
#define SET_BIT(BYTE, BIT)  ((BYTE) |=  (1 << (BIT))) // Bit Zustand in Byte setzen
#define CLR_BIT(BYTE, BIT)  ((BYTE) &= ~(1 << (BIT))) // Bit Zustand in Byte loeschen
#define TGL_BIT(BYTE, BIT)  ((BYTE) ^=  (1 << (BIT))) // Bit Zustand in Byte wechseln (toggle)

//Makros fr die verschiedenen I2C Befehle 
#define DO_TWCR_ACK   TWCR = (1<<TWEN)|(1<<TWIE)|(1<<TWINT)|(1<<TWEA)|(0<<TWSTA)|(0<<TWSTO)|(0<<TWWC);	//ACK nach empfangenen Daten senden/ ACK nach gesendeten Daten erwarten
#define DO_TWCR_NACK  TWCR = (1<<TWEN)|(1<<TWIE)|(1<<TWINT)|(0<<TWEA)|(0<<TWSTA)|(0<<TWSTO)|(0<<TWWC);	//NACK nach empfangenen Daten senden/ NACK nach gesendeten Daten erwarten
#define DO_TWCR_RESET TWCR = (1<<TWEN)|(1<<TWIE)|(1<<TWINT)|(1<<TWEA)|(0<<TWSTA)|(1<<TWSTO)|(0<<TWWC);	//switch to the non adressed slave mode...

#define PRESCALER_VAL       60      // Faktor Vorteiler = 60
#define CYCLE10MS_MAX       10      // Faktor Hundertstel = 10
#define CYCLE100MS_MAX      10      // Faktor Zehntel = 10

#define TWI_BUFFERSIZE_RX		2		// TWI_BUFFERSIZE fr empfangende Daten, Zahl entspricht der Anzahl der empfangenden Register (max 254)
#define TWI_BUFFERSIZE_TX			1		// TWI_BUFFERSIZE fr zu sendende Daten, Zahl entspricht der Anzahl der zu sendende Daten (max 254)
#define ASC_NULL            0x30    // Das Zeichen '0' in ASCII
#define ASC_FULL_STOP       0x2E    // Das Zeichen ':' in ASCII
 
unsigned char softwarePrescaler = PRESCALER_VAL;    // Zaehlvariable Vorteiler
unsigned char cycle10msCount    = CYCLE10MS_MAX;    // Zaehlvariable Hundertstel
unsigned char cycle100msCount   = CYCLE100MS_MAX;   // Zaehlvariable Zehntel

bool timertick;                     // Bit-Botschaft alle 0,166ms (bei Timer-Interrupt)
bool cycle10msActive;               // Bit-Botschaft alle 10ms
bool cycle100msActive;              // Bit-Botschaft alle 100ms
bool cycle1sActive;                 // Bit-Botschaft alle 1s
bool TWI_Received;

int     tValue				= 0;    // Variable fuer die Temperatur (in 1/10 C)
int     tValueMax			=-300;  // Variable fuer maximale Temperatur (1/10 C)

uint8_t				TWI_Address         =  0b0001010;
uint8_t				TWI_AddressMask     = 0b11111110;
uint8_t				TWI_Register		= 0b00000000;
volatile uint8_t	TWI_BufferPosRX; //"Adressregister" fr den Buffer
volatile uint8_t	TWI_BufferPosTX; //"Adressregister" fr den Buffer
volatile uint8_t	TWI_DataRX[TWI_BUFFERSIZE_RX];
volatile uint8_t	TWI_DataTX[TWI_BUFFERSIZE_TX];

//Funktionsprototypen
void initTimer0(void);
void initDisplay(void);
void refreshDisplayTemp(int tempValue, char line, char pos);
void refreshDisplay(void);

void initTwi(void);
void setTwiAddress(char Address);

int main(void)
{
    initDisplay();						// Initialisierung LCD-Anzeige
    initTimer0();						// Initialisierung von Timer0
    setTwiAddress(TWI_Address);			// eigene Adresse setzen
    DDRB=0xF0; 
    while (1) 
    {
		if(TWI_Received)
		{
			tValue = (int) (TWI_DataRX[0] | (TWI_DataRX[1]<<8));           // Daten an PortC ausgeben			
			if( tValue	   >=tValueMax)			// aktueller Wert mit Maximalwert
				tValueMax  = tValue;			// vergleichen und ggf. ersetzen
		}
        if(cycle100msActive)				// Durchfuehrung der Funktion einmal pro 100ms
        {
	        cycle100msActive = 0;				// Taktbotschaft zuruecksetzen
			TWI_Received = 0;					// TWI Botschaft zuruecksetzen
			TWI_DataTX[0]=PORTB;
			refreshDisplay();
        }
    } 
}

// Timer Initialisierung ==============================================================
//
// Initialisierung des Timer0 zur Erzeugung eines getakteten Interrupts.
// Er dient dazu, die benoetigten Taktbotschaften zu erzeugen.
void initTimer0()
{
	TCCR0A  |= (0<<WGM00)
			|  (0<<WGM01);            // Timer 0 auf "Normal Mode" schalten
	TCCR0B  |= (0<<WGM02)
			|  (1<<CS01 );            // mit Prescaler /8 betreiben
	TIMSK0  |= (1<<TOIE0);            // Overflow-Interrupt aktivieren
}

// Interrupt-Routine ==========================================================
//
ISR (TIMER0_OVF_vect)
/*  In der Interrupt-Routine sind die Softwareteiler realisiert, die die Takt-
    botschaften (10ms, 100ms, 1s) fuer die gesamte Uhr erzeugen. Die Interrupts
    werden von Timer 0 ausgeloest (Interrupt Nr. 1)
  
*/
{
    timertick = 1;                  // Botschaft 0,166ms senden
    --softwarePrescaler;			// Vorteiler dekrementieren
    if (softwarePrescaler==0)       // wenn 0 erreicht: 10ms abgelaufen
    {
        softwarePrescaler = PRESCALER_VAL;	//    Vorteiler auf Startwert
        cycle10msActive = 1;				//    Botschaft 10ms senden
        --cycle10msCount;					//    Hunderstelzaehler dekrementieren
  
        if (cycle10msCount==0)				// wenn 0 erreicht: 100ms abgelaufen
        {
            cycle10msCount = CYCLE10MS_MAX; // Teiler auf Startwert
            cycle100msActive = 1;			//    Botschaft 100ms senden
            --cycle100msCount;              //    Zehntelzaehler dekrementieren
  
            if (cycle100msCount==0)         // wenn 0 erreicht: 1s abgelaufen
            {
                cycle100msCount = CYCLE100MS_MAX;	//    Teiler auf Startwert
                cycle1sActive = 1;					//    Botschaft 1s senden
            }
        }
    }
}

// TWI Adresse ==============================================================
//
// Setzen der I2C Adresse auf die der Slave hrt
void setTwiAddress(char Address)                       
{
    TWAR = (Address<<1);								// Adresse in das Pseudoregister schreiben
    TWAMR= TWI_AddressMask;                             // Adressmaske in das Pseudoregister schreiben
	TWCR &= ~(1<<TWSTA)|(1<<TWSTO);
    TWCR =   (1<<TWEA)|(1<<TWEN)|(1<<TWIE);				// Enable Ack, Enable Interupt und Enable TWI
	TWI_BufferPosRX=0xFF;									// Buffer Position zurcksetzen 
	TWI_BufferPosTX=0xFF;
	sei();
}

void initDisplay()                
{
	lcd_init();											// Initialisierungsroutine aus der lcd_lib
	
	lcd_displayMessage("- Experiment 8c-",0,0);			// Ausgabe in erster Zeile
	lcd_displayMessage(" Temperature I2C",1,0);			// Ausgabe in zweiter Zeile
	
	_delay_ms(1000);									// Wartezeit nach Initialisierung
	
	lcd_displayMessage("Temp.         C",0,0);			// Ausgabe in erster Zeile
	lcd_displayMessage("Maximum       C",1,0);			// Ausgabe in zweiter Zeile
														// "C" wird als C dargestellt
}                                  

// TWI ISR ==============================================================
//
// ISR, die bei einem Ereignis auf dem Bus ausgelst wird. Im Register TWSR befindet sich dann
// ein Statuscode, anhand dessen die Situation festgestellt werden kann.
// Basierend auf: https://rn-wissen.de/wiki/index.php/TWI_Slave_mit_avr-gcc

ISR (TWI_vect)
{
	uint8_t data=0;
	switch (TW_STATUS) //TWI-Statusregister prfen und ntige Aktion bestimmen
	{	// Slave Receiver
		case TW_SR_SLA_ACK:						// 0x60 Slave Receiver, Slave wurde adressiert
			DO_TWCR_ACK;								//nchstes Datenbyte empfangen, ACK danach senden
			TWI_BufferPosRX=0xFF;						//Bufferposition ist undefiniert
			break;
		
		case TW_SR_DATA_ACK:					// 0x80 Slave Receiver, ein Datenbyte wurde empfangen
			data=TWDR;								//Empfangene Daten auslesen
			if (TWI_BufferPosRX == 0xFF)					//erster Zugriff, Bufferposition setzen
			{
				if(data<TWI_BUFFERSIZE_RX+1)				//Kontrolle ob gewnschte Adresse im erlaubten bereich
					TWI_BufferPosRX= data;					//Bufferposition wie adressiert setzen
				else
					TWI_BufferPosRX=0;						//Adresse auf Null setzen. Ist das sinnvoll? TO DO!
				DO_TWCR_ACK;								// nchstes Datenbyte empfangen, ACK danach, um nchstes Byte anzufordern
			}
			else									//weiterer Zugriff, nachdem die Position im Buffer gesetzt wurde. Nun die Daten empfangen und speichern
			{			
				if(TWI_BufferPosRX<TWI_BUFFERSIZE_RX+1)
				{
					TWI_DataRX[TWI_BufferPosRX]=data;			//Daten in Buffer schreiben
					if( TWI_BufferPosRX == TWI_BUFFERSIZE_RX-1)		// Wenn Ende des Buffers erreicht
						TWI_Received = 1;						// Daten-Empfangen Flag setzen
				}
				TWI_BufferPosRX++;						//Buffer-Adresse weiterzhlen fr nchsten Schreibzugriff
				DO_TWCR_ACK;
			}
			break;
			
		//Slave transmitter
		case TW_ST_SLA_ACK:							//0xA8 Slave wurde im Lesemodus adressiert und hat ein ACK zurckgegeben 
		case TW_ST_DATA_ACK:						//oder 0xB8 Slave Transmitter, Daten wurden angefordert
			if (TWI_BufferPosTX == 0xFF)					//zuvor keine Leseadresse angegeben!
				TWI_BufferPosTX=0;
			if(TWI_BufferPosTX<TWI_BUFFERSIZE_TX+1)
			{
				TWDR = 0xFF;			//Datenbyte senden
				TWI_BufferPosTX++;						//bufferadresse fr nchstes Byte weiterzhlen
				TGL_BIT(PORTB, 4);				
			}
			else
			{
				TWDR=0;								//Kein Daten mehr im Buffer
				TWI_BufferPosTX = 0xFF;
			}
			DO_TWCR_ACK;
			break;
		case TW_SR_STOP:
			DO_TWCR_ACK;
			break;
		case TW_ST_DATA_NACK: // 0xC0 Keine Daten mehr gefordert,
		case TW_SR_DATA_NACK: // ODER 0x88 
		case TW_ST_LAST_DATA: // ODER 0xC8  Last data byte in TWDR has been transmitted (TWEA = 0); ACK has been received
		default:
			DO_TWCR_RESET;
			break;
	} 
}

// Anzeigefunktion ==============================================================
//
// Der aktuelle Temperatur und die maximale Temperatur werden ausgegeben
void refreshDisplay()
{
	refreshDisplayTemp(tValue,      0, 9);      // aktuelle Temperatur ab Position 0,9
	refreshDisplayTemp(tValueMax,   1, 9);      // maximale Temperatur ab Position 1,9
}

// Anzeigetreiber fuer Temperaturanzeige ==============================================================
//
// Beschreiben der Anzeige mit dem erstellten Temperaturwert
// und mit dem maximalen Wert (wird alle 1 s aufgerufen).
//
// Umrechnung der Zahlenwerte (1/10 C) in Anzeigewerte wie folgt:
// Hunderter: einfache Integer-Teilung (/100).
// Zehner: Modulo-Ermittlung (%100), d.h. Rest bei der Teilung durch 100
//         dann nochmals Integer-Teilung (/10) dieses Restes.
// Einer:  Modulo-Ermittlung (%10), d.h. Rest bei der Teilung durch 10.
//
// Umrechnung in ASCII-Werte fuer die Anzeige durch Addition von 0x30.
void refreshDisplayTemp(int tempValue, char line, char pos)
{
	lcd_gotoxy(line, pos);                  // Startposition fuer Temperatur-Wert
	if (tempValue>=0)                        // zuerst Vorzeichen: ' ' oder '-'
	{
		lcd_putc(' ');
	}
	else
	{
		lcd_putc('-');
		tempValue = -tempValue;             // Vorzeichenumkehr bei negativer Zahl
	}
	lcd_putc   (tempValue/100 + ASC_NULL);  // Hunderter ausgeben (C Zehner)
	tempValue = tempValue%100;
	lcd_putc   (tempValue/10  + ASC_NULL);  // Zehner ausgeben (C Einer)
	lcd_putc   (ASC_FULL_STOP);             // Punkt ausgeben
	lcd_putc   (tempValue%10  + ASC_NULL);  // Einer ausgeben (C Zehntel)
}
